/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY.                         *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.  Contact information: bergmark@cs.cornell.edu     *
 ******************************************************************************/
package cnrg.itx.ds;

import cnrg.itx.ds.ArrayRecords;

/**
 * This class is an inteface to the java native methods talking to the directory server.  User should not directly call any methods of this class. <br>
 * Alternatively, if user prefers to provide his/her directory database server, he/she can define DSComm-derived class and the directory server. <br><p>
 * Example below shows myDSComm class derived from DSComm : <br><p>
 * 
 *  import cnrg.itx.ds.*; <br>
 *  public class myDSComm extends DSComm <br>
 *  { <br>
 *      ... <br>
 *		private String USERID2EXT_DOMAIN = "myDomainName"; <br>
 *		... <br>
 *		public String getUserID2ExtDomain() { <br>
 *			return USERID2EXT_DOMAIN; <br>
 *		} <br>
 *		public void setConfigPath(byte[] filepath) { <br>
 *			// your customized code here <br>
 *		} <br>
 *	
 *		public ArrayRecords getRecord(byte[] key) { <br>
 *			// your customized code here <br>
 *		} <br>
 *	
 *		public void deleteRecord(byte[] key, byte[] rec){ <br>
 *			//  your customized code here <br>
 *		} <br>
 *	
 *		public void addRecord(byte[] key, byte[] rec){ <br>
 *			//  your customized code here <br>
 *		} <br>
 *		... <br>
 *	} <br>
 */
public class DSComm
{
	/**
	 * Directory database doamin for storing extension distribution entries. <br>
	 * 7 sub-domains are defined for extension distribution (e.g. "ds.extdistrib4.itx.cnrg." for storing 4-digit extension distribution entries) 
	 */ 
	private String PREFIX_EXTDISTRIB_DOMAIN = "ds.extdistrib";
	private int MIN_INFIX_EXTDISTRIB = 1;	// min. 1-digit extension
	private int MAX_INFIX_EXTDISTRIB = 7;	// max. 7-digit extension
	private String POSTFIX_EXTDISTRIB_DOMAIN = ".itx.cnrg.";	
	/**
	 * Directory database domain for storing userid-to-authentication entries
	 */
	private String USERID2SEC_DOMAIN = ".userid.to.sec.";
	/**
	 * Directory database domain for storing userid-to-extension entries
	 */	
	private String USERID2EXT_DOMAIN = ".userid.to.ext.";
	/**
	 * Directory database domain for storing extension-to-userRecord entries
	 */
	private String EXT2USER_DOMAIN = ".ext.to.user.";
	/**
	 * Directory database domain for storing extension-to-roamingLocation entries
	 */
	private String EXT2ROAMLOC_DOMAIN = ".ext.to.roamloc.";
	/**
	 * Directory database domain for storing extension-to-dynamicLocationList entries
	 */
	private String EXT2DYNAMICLOCLIST_DOMAIN = ".ext.to.dynamicloclist.";
	/**
	 * Directory database domain for storing extension-to-defaultLocationList entries
	 */
	private String EXT2DEFAULTLOCLIST_DOMAIN = ".ext.to.defaultloclist.";
	/**
	 * Directory database domain for storing extension-to-customLocationList entries
	 */
	private String EXT2CUSTOMLOCLIST_DOMAIN = ".ext.to.customloclist.";
	/**
	 * Directory database domain for storing customLocation entries
	 */
	private String CUSTOMLOC_DOMAIN = ".custom.loc.";
	/**
	 * Gatewaysrv username in the ITX network
	 */
	private String GATEWAY_USERID = "gatewaysrv";
	
	
	/**
	 * Gets directory database doamin for storing extension distribution entries.
	 * @return String object to represent the domain name
	 */
	public String getExtDistribDomain(int nSubDomain) {
		return PREFIX_EXTDISTRIB_DOMAIN+String.valueOf(nSubDomain)+POSTFIX_EXTDISTRIB_DOMAIN;
	}
	
	/**
	 * Gets lowest sub-domain index of the extension distribution domain
	 * @return int value to represent the index
	 */
	public int getMinSubDomain() {
		return MIN_INFIX_EXTDISTRIB;
	}
	
	/**
	 * Gets largest sub-domain index of the extension distribution domain
	 * @return int value to represent the index
	 */
	public int getMaxSubDomain() {
		return MAX_INFIX_EXTDISTRIB;
	}
	
	/**
	 * Gets directory database domain for storing userid-to-extension entries.
	 * @return String object to represent the domain name
	 */
	public String getUserID2SecDomain() {
		return USERID2SEC_DOMAIN;
	}
	
	/**
	 * Gets directory database domain for storing userid-to-extension entries
	 * @return String object to represent the domain name
	 */
	public String getUserID2ExtDomain() {
		return USERID2EXT_DOMAIN;
	}
	
	/**
	 * Gets directory database domain for storing extension-to-userRecord entries
	 * @return String object to represent the domain name
	 */
	public String getExt2UserDomain() {
		return EXT2USER_DOMAIN;
	}
	
	/**
	 * Gets directory database domain for storing extension-to-roamingLocation entries
	 * @return String object to represent the domain name
	 */
	public String getExt2RoamLocDomain() {
		return EXT2ROAMLOC_DOMAIN;
	}
	
	/**
	 * Gets directory database domain for storing extension-to-dynamicLocationList entries
	 * @return String object to represent the domain name
	 */
	public String getExt2DynamicLocListDomain() {
		return EXT2DYNAMICLOCLIST_DOMAIN;
	}
	
	/**
	 * Gets directory database domain for storing extension-to-defaultLocationList entries
	 * @return String object to represent the domain name
	 */
	public String getExt2DefaultLocListDomain() {
		return EXT2DEFAULTLOCLIST_DOMAIN;
	}
	
	/**
	 * Gets directory database domain for storing extension-to-customLocationList entries
	 * @return String object to represent the domain name
	 */
	public String getExt2CustomLocListDomain() {
		return EXT2CUSTOMLOCLIST_DOMAIN;
	}
	
	/**
	 * Gets directory database domain for storing customLocation entries
	 * @return String object to represent the domain name
	 */
	public String getCustomLocDomain() {
		return CUSTOMLOC_DOMAIN;
	}
	
	/**
	 * Gets the gatewaysrv username in the ITX network
	 * @return String object to represent the username
	 */
	public String getGatewayUserID() {
		return GATEWAY_USERID;
	}
	
	/**
	 * Set the config filepath in the DSComm.dll to be able to communicate with the directory server
	 * @param filepath a file path of the config file (resolv.conf)
	 */
	public native synchronized void setConfigPath(byte[] filepath);
	
	/**
	 * Get record(s) associated with the domain entry.  A domain entry can be associated with 1 or more than 1 record. <br>
	 * For example, in the domain, "ext.to.user.", an entry, "10011.ext.to.user." has 1 and only 1 record, which is "userid=mr85@cornell.edu;custmsg=". <br>
	 * For example, in the domain, "ext.to.defaultloclist.", an entry, "10011.ext.to.defaultloclist." can have more than 1 record. <br>
	 * @param key domain entry (e.g. "10011.ext.to.user.")
	 * @return record(s) (associated with the entry) to be returned (e.g. "userid=mr85@cornell.edu;custmsg=").  If no record found, it returns an empty ArrayRecords object.
	 */ 
	public native synchronized ArrayRecords getRecord(byte[] key);
	
	/**
	 * Delete a record (or all records) associated with the domain entry
	 * @param key domain entry 
	 * @param rec an existing record (associated with the entry) to be deleted.  If rec is null, it deletes all records associated with the domain entry.
	 */
	public native synchronized void deleteRecord(byte[] key, byte[] rec);
	
	/**
	 * Add an record associated with the domain entry
	 * @param key domain entry
	 * @param rec a record (associated with the entry) to be added.  If the record already exists in the directory database, the input record will be ignored.
	 */
	public native synchronized void addRecord(byte[] key, byte[] rec);

	static {
	  System.loadLibrary("DSComm");	// DSComm.dll in windows platform  (libDSComm.so equivalent in unix platform)
	}
}
